------------------------------------------------------------------------------
-- Import Boujou Track, Revision: 2.0 
--
-- tool script
--
-- This script will import 2D Boujou ASCII tracks. When executed a dialog will appear
-- asking for the path to the Boujou ASCII file. Fusion will read the entire file, 
-- and present a dialog with dropdowns listing all of the potential tracks contained 
-- in the file, and all positional controls in the current tool. Clicking OK on this 
-- dialog will apply the selected track to the selected control.
-- 
-- If existing path of any sort is already present on the control a dialog will 
-- ask confirmation to proceed. The keyframes from the ASCII track are merged into the 
-- existing path, so if frames outside the time described in the Boujou track already 
-- exist they will be preserved
--
-- Known Problems
-- None
--
-- written by : Isaac Guenard (izyk@eyeonline.com)
-- written    : August 27th, 2003
-- updated : Sept 27, 2005
-- changes : updated for 5
------------------------------------------------------------------------------

-----------------------------------------------------
-- wish list
-- option to overwrite existing connection rather 
--   than reuse if animation is found


-----------------------------------------------------
-- trim(strTrim)
-- 
-- returns strTrim with leading and trailing spaces 
-- removed.
-----------------------------------------------------



function trim(strTrim)
	strTrim = string.gsub(strTrim, "^(%s+)", "") -- remove leading spaces
	strTrim = string.gsub(strTrim, "$(%s+)", "") -- remove trailing spaces
	return strTrim
end

function handled(name, frame, x, y) 	
	if not (tracks[name]) then tracks[name] = {} end 
	tracks[name][frame] = {x, y}
end


---[[ MAIN ]]--------------------------------------------------

-- running as tool script?
if (composition == nil) or (tool == nil) then
	print("This script is a tool script. Place the script in the DFusion:\Tools directory and run it from a tools context menu.")
	os.exit()
end

-- build a table of controls with the point (positional) datatype
controls = {}
opt_controls = {}

inps = tool:GetInputList()
for i, inp in pairs(inps) do
	inpa = inp:GetAttrs() 
	if inpa.INPS_DataType == "Point" then 
		table.insert(controls, inpa.INPS_Name)
		table.insert(opt_controls, inp)
	end
end

-- complain if none of the controls have this datatype
if table.getn(controls) == 0 then
	ret = composition:AskUser("Control is animated!", {
		{"warning", Name = "warning", "Text", Default="The selected tool does not contain any co-ordinate type controls. This script imports a Boujou ASCII  path to a co-ordinate or positional type control.", Wrap=true, Lines=8}
		})
	return
end


while track == nil do
	-- ask the user what Boujou file to process
	dialog = {}
	if ret == nil then
		table.insert(dialog, {"Filename", "FileBrowse", Default = ""})
	else
		table.insert(dialog, {"Filename", "FileBrowse", Default = ret.Filename})
	end
	
	if err then
		table.insert(dialog, {"warning", Name = "warning", "Text", Default=err, Wrap=true, Lines=8})
	end
	
	ret = composition:AskUser("Select a Boujou ASCII Path", dialog)
	if ret == nil then return end
	
	-- can we open the file
	tracks = {}
	
	track, err = io.open(MapPath(ret.Filename), "r")
end


-- this portion 

line = track:read("*l")
while line do
	line = trim(line)

	if string.sub(line, 1, 1) ~= "#" then
		string.gsub(line, "([%S]+)%s*([%S]+)%s*([%S]+)%s*([%S]+)", handled)
	end
	line = track:read("*l")
end
track:close()
track = nil

options = {}
for i, v in pairs(tracks) do
	table.insert(options, i)
end

ret = composition:AskUser("Select a Track", {
	{"track", Name = "Select a Track", "Dropdown", Options=options},
	{"control", Name = "Apply Track To", "Dropdown", Options=controls}
	})

if ret == nil then return end

-- select a track and a control
track = tracks[options[ret.track+1]]

if ret == nil then return end
control = opt_controls[ret.control+1]

attrs = tool:GetAttrs()
if attrs.TOOLI_ImageWidth == nil then
	pref = composition:GetPrefs()
	-- this tool has never rendered
	ret = composition:AskUser("Unknown Image Size!", {
		{"warning", Name = "warning", "Text", Default="The selected tool has never rendered, so the size of the image it processess cannot be determined.\n\n"..
		"Please enter the width and height of the image below, and select OK to proceed (or select CANCEL to exit).", Wrap=true, Lines=8},
		{"width", "Slider", Integer=true, Default=pref.Comp.FrameFormat.Width, Min=2, Max=4096},
		{"height", "Slider", Integer=true, Default=pref.Comp.FrameFormat.Height, Min=2, Max=4096}
		})
	if ret == nil then return end
	imagex = ret.width
	imagey = ret.height
else
	imagex = attrs.TOOLI_ImageWidth
	imagey = attrs.TOOLI_ImageHeight
end

composition:StartUndo("Import Boujou Track")

-- is the control animated?
if control:GetAttrs().INPB_Connected == false then
	control = Path({})
else
	ret = composition:AskUser("Control is animated!", {
		{"warning", Name = "warning", "Text", Default="The selected control is already animated or connected to another control."..
		"\n\nSelect OK to continue anyway, or CANCEL to abort. If you select OK the keys in the track file will be merged with existing keys.", Wrap=true, Lines=10}
		})
	if ret == nil then return end
end

--finally do the keyframes
if composition.UpdateMode then
	if composition.UpdateMode == "None" or composition.UpdateMode == "Some" then
		hold = composition.UpdateMode
		composition.UpdateMode = "All"
	end
end

for frame, pos in pairs(track) do
	control[tonumber(frame)] = {tonumber(pos[1]) / imagex, 1.0 - (tonumber(pos[2]) / imagey), 0}
end

composition:EndUndo(true)

if hold then
	composition.UpdateMode = hold
end
